﻿<#

.SYNOPSIS
    Name:           <script.ps1>
    Summary:        <Brief synopsis>

.DESCRIPTION
    <Brief description of script>

.PARAMETER <Parameter _Name>
    <Brief description of parameter input required. Repeat this attribute if required>

.INPUTS
    <Inputs if any, otherwise state None>

.OUTPUTS
    <Outputs if any, otherwise state None - example: Log file stored in C:\Windows\Temp\<name>.log>

.NOTES
    Version:        1.0 
    Author:         Jerald Bottcher
    Creation Date:  20190205
    Purpose/Change: Rapid Inspection 2019 v 11.0.1.0

.COMMON SCRIPT SYNTAX
    # Traditional MSI installation

        & "$SysFolder32\msiexec.exe" /i "$ScriptPath\installer.msi" TRANSFORMS="msitranform.mst" /qb! REBOOT=REALLYSUPPRESS ALLUSERS=1 | Out-Null

    # Traditional EXE installation

        & "$ScriptPath\npp6.6.9.installer.exe" /S
        Get-Process | Where {$_.Name -Eq "npp6.6.9.installer"}) | Wait-Process
       
.EXAMPLE
    powershell.exe -noprofile -file ".\Verb-Noun.ps1"
                     
#>

#---------------------------------------------------------[Initializations]--------------------------------------------------------

#Import-Module <ModuleName>

#----------------------------------------------------------[Declarations]----------------------------------------------------------

# Public Desktop (All Users Desktop) - C:\Users\Public\Desktop
$PubDesktop =  (New-Object -com shell.application ).namespace(25).self.path

# Public Start Menu - (All Users Start Menu) - C:\ProgramData\Microsoft\Windows\Start Menu\Programs
$PubStart =  (New-Object -com shell.application ).namespace(23).self.path

# Program Files - c:\Program Files
$PFiles =  (New-Object -com shell.application ).namespace(38).self.path

# Program Files - c:\Program Files (x86)
$PFiles86 =  (New-Object -com shell.application ).namespace(42).self.path

# Windows - c:\windows
$Win = (New-Object -com shell.application ).namespace(36).self.path

# System 32 - c:\windows\system32
$SysFolder64 = (New-Object -com shell.application ).namespace(37).self.path

# SysWOW64 - c:\windows\syswow64
$SysFolder32 = (New-Object -com shell.application ).namespace(41).self.path

#System Drive - c:
$SYSDrive = $env:SystemDrive

#Path that the script is located in.
$ScriptPath = Split-Path $script:MyInvocation.MyCommand.Path
# Account for inconsistency with $ScriptPath. When the $ScriptPath resolves to a drive
# letter it adds a backslash that isn't normally there.
if ($ScriptPath.EndsWith("\"))
{
	$ScriptPath = $ScriptPath.TrimEnd("\")
}

#-----------------------------------------------------------[Functions]------------------------------------------------------------

<#

.SYNOPSIS
    Name:           Check-LastExitCode
    Summary:        Checks the $LastExitVariable for success and exits if unsuccessful.

.DESCRIPTION
    $LastExitCode (a PowerShell automatic variable) contains the exit code of the last Windows-based program that was run.
        
    This function checks $LastExitVariable for success and exits if unsuccessful.
	By default a return value of 0 is considered successful. Customization of the
	successful exit codes can be accomplished by modifying the IF statement.
       
    Run the following command for more information:  
        Get-Help about_Automatic_Variables -ShowWindow

.PARAMETERS
    None.

.INPUTS
    None.

.OUTPUTS
    Quit Script with error code if $LastExitCode returns an unwanted result.

.NOTES
    Version:        1.1 
    Author:         MNIT
    Creation Date:  20090101 
    Purpose/Change: The function (previously named CheckIt) was renamed in v1.1 (2016)

.EXAMPLE
    Check-LastExitCode
                           
#>

function Check-LastExitCode
{
	if ($LastExitCode -ne 0){
		$Host.SetShouldExit($LastExitCode)
		Write-Host $LastExitCode
        Start-Sleep -s 10
		Exit
	}
}

<#

.SYNOPSIS
    Name:           Create-Shortcut
    Summary:        Creates a shortcut.

.DESCRIPTION
    This function accepts a PS Object and creates a shortcut based on parameters set within the object.

.PARAMETER lnk
    Mandatory:      True
    Position:       0
    Alias:          Shortcut

.INPUTS
    This function accepts a PSObject with the following parameters set:
        path                -Destination Path with lnk name
        target              -Target of the lnk
        working_directory   -Working Directory of the lnk
        description         -lnk description
        arguments           -lnk arguments
        icon_location       -Path filename of ico file

.OUTPUTS
    None.

.NOTES
    Version:        1.0 
    Author:         MNIT
    Creation Date:  20090101 
    Purpose/Change: Initial function development

.EXAMPLE
    $lnkobject = New-Object PSObject -property @{
        path = "$PubDesktop\test.lnk"
        target = "$SysFolder64\notepad.exe"
        working_directory = $SysFolder64
        description = "THIS IS A TEST"
        arguments = "$SYSDrive\cc.txt"
        icon_location = "$SYSDrive\rotate1.ico"
    }

    Create-Shortcut -Shortcut $lnkobject   
                              
#>

function Create-Shortcut
{
    Param(
        [CmdletBinding()]
        [parameter(Mandatory = $true, Position = 0)]
        [Alias('Shortcut')]
        [psobject]
        $lnk
    )
    
    # Use Split Path to split the LNK file name off of $ar1 to test if the shortcut parent folder exists.
    # Split Path at each \ in the path.
    $pathpieces = $lnk.path.Split("\")
    
    # Get the filename from the array created by splitting the path. The file name will always be the last array element.
    $filename = $pathpieces[$pathpieces.count - 1]
    
    # Trim the filename from the end of the LNK Path
    $parentfolder = $lnk.path.TrimEnd($filename)
    
    # Check existence of the shortcut parent folder
    if (!(Test-Path $parentfolder))
    {
        New-Item ($parentfolder) -type directory
    }
    
    # Create shortcut based on the function parameters
    $wshell = New-Object -comObject WScript.Shell
    $link = $wshell.CreateShortcut($lnk.path)		
    $link.TargetPath = $lnk.target
    $link.WorkingDirectory = $lnk.working_directory	
    $link.Description = $lnk.description
    $link.Arguments = $lnk.arguments
    $link.IconLocation = $lnk.icon_location		
    $link.Save()
}

<#

.SYNOPSIS
    Name:           Set-ActiveSetup
    Summary:        Adds a value to the Active setup area of the registry

.DESCRIPTION
    Adds entry to the ActiveSetup location within the Windows Registry 
    to execute against every user profile that logs into the system 
    (will run once per user profile)

.PARAMETER ActiveSetupUniqueName 
    Mandatory:      True
    Position:       0
    Alias:          Name

.PARAMETER ActiveSetupStubPath
    Mandatory:      True
    Position:       1
    Alias:          Stubpath

.PARAMETER ActiveSetupVersion
    Mandatory:      True
    Position:       2
    Alias:          Version

.INPUTS
    This function requires that 3 string values are passed as parameters

.OUTPUTS
    None.

.NOTES
    Version:        1.1 
    Author:         MNIT
    Creation Date:  20150101
    Purpose/Change: Modified parameter design and renamed function (v 1.1)

.EXAMPLE
    $name = "launchnotepad"
    $stubpath = "notepad" 
    $version = "1"
    Set-ActiveSetup -Name $name -Stubpath $stubpath -Version $version
                           
#>

function Set-ActiveSetup
{
    param(
        [CmdletBinding()]
        [parameter(Mandatory = $true, Position = 0)]
        [Alias('Name')]
        [string]
        $ActiveSetupUniqueName,
	    [parameter(Mandatory = $true, Position = 1)]
        [Alias('Stubpath')]
        [string]
        $ActiveSetupStubPath,
	    [parameter(Mandatory = $true, Position = 2)]
        [Alias('Version')]
        [string]
        $ActiveSetupVersion
    )

    $ParentKey="HKLM:Software\Microsoft\Active Setup\Installed Components"
  
    $Key=$ParentKey + "\" + $ActiveSetupUniqueName
    if (!(Test-Path $Key))
    {
        New-Item -type Directory $($ParentKey + "\" + $ActiveSetupUniqueName)
    }
    else 
    {
        Write-Host "Key exists"
    }

    Set-ItemProperty $($Key) -Name "StubPath" -Value $ActiveSetupStubPath
    Set-ItemProperty $($Key) -Name "Version" -Value $ActiveSetupVersion
}


<#

.SYNOPSIS
    Name:           Uninstall-MSIApplication
    Summary:        Uninstall applications based on a lists of MSI product codes in a text file

.DESCRIPTION
    Accepts an array of MSI product codes as a parameter and uninstalls the corresponding applications if they are discovered on a machine

.PARAMETER ProductCodes
    Mandatory:      True
    Position:       0
    Alias:          Guids

.INPUTS
    An array of MSI product codes

.OUTPUTS
    None.

.NOTES
    Version:        1.1 
    Author:         MNIT
    Creation Date:  20150101
    Purpose/Change: Modified parameter design and renamed function (v 1.1)

.EXAMPLE 
    #Add MSI product code GUIDS that you want uninstalled into a text file named GUIDS.txt
    $guids = Get-Content $ScriptPath\Guids.txt
    Uninstall-MSIApplication -ProductCodes $guids
                           
#>

function Uninstall-MSIApplication
{
    param(
        [CmdletBinding()]
        [parameter(Mandatory = $true, Position = 0)]
        [Alias('Guids')]
        [string[]]
        $ProductCodes
    )
    foreach ($p in $ProductCodes){ 
        # Uninstall Guid
        if (Test-Path "HKLM:Software\Microsoft\Windows\CurrentVersion\Uninstall\$p"){
	        & "$Win\system32\msiexec.exe" /x "$p" /qb! REBOOT=ReallySuppress | Out-Null
        }
        elseif (Test-Path "HKLM:Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\$p"){
	        & "$Win\system32\msiexec.exe" /x "$p" /qb! REBOOT=ReallySuppress | Out-Null
        }
    }
}

<#

.SYNOPSIS
    Name:           Kill-Process
    Summary:        Loop through an array of processes, stopping each process

.DESCRIPTION
    Accepts an array of process names as a parameter.  Loops through the array, killing each process

.PARAMETER Processes
    Mandatory:      True
    Position:       0
    Alias:          Procs

.INPUTS
    An array of process names

.OUTPUTS
    None.

.NOTES
    Version:        1.1 
    Author:         MNIT
    Creation Date:  20150101
    Purpose/Change: Modified parameter design and renamed function (v 1.1)

.EXAMPLE 1 
    $Procs = "notepad", "Winword"
    Kill-Process -processes $procs
    
.EXAMPLE 2
    #Add list of processes to a text file named KillProcesses.txt
    $Procs = Get-Content "$ScriptPath\KillProcesses.txt"
    Kill-Process -processes $procs
                           
#>

function Kill-Process
{
    param(
        [CmdletBinding()]
        [parameter(Mandatory = $true, Position = 0)]
        [Alias('procs')]
        [string[]]
        $Processes
    )

    foreach ($p in $Processes){ 
        Get-Process | Where-Object { $_.Name -Eq $p } | Stop-Process -force
    }
}


#-----------------------------------------------------------[Execution]------------------------------------------------------------


#####################################
# Prepping for Uninstall            #
#####################################
& "$SysFolder32\msiexec.exe" /i "$ScriptPath\Rapid Inspection 2019 v11.0.1.0.msi" /qb! TRANSFORMS="$ScriptPath\RI2019.mst" | Out-Null

#####################################
# UnInstalling                      #
#####################################

#$guids = Get-Content $ScriptPath\Guids.txt
#Uninstall-MSIApplication -ProductCodes $guids

#####################################
# Installing                        #
#####################################




#####################################
# Cleanup                           #
#####################################